'use strict';

/* --------------------------------------------------------------
 init_class_fixes.js 2017-05-30 
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Initialize Class Fixes
 *
 * This module must set as many compatibility classes as possible. Wherever it is
 * certain that an HTML class will be present it must be automatically set by this
 * module.
 *
 * @module Compatibility/init_class_fixes
 */
gx.compatibility.module('init_class_fixes', ['url_arguments'],

/**  @lends module:Compatibility/init_class_fixes */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Callbacks for checking common patterns.
     *
     * @var {array}
     */
    fixes = [],


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // OPERATIONS
    // ------------------------------------------------------------------------

    /**
     * Add gx-compatibility class to body element.
     */
    fixes.push(function () {
        if (!$('body').hasClass('gx-compatibility')) {
            $('body').addClass('gx-compatibility');
        }
    });

    /**
     * Add the gx-container custom predefined selectors.
     */
    fixes.push(function () {
        // Append the following array with extra custom selectors.
        var customSelectors = ['.dataTableRow', '.dataTableHeadingRow', '.dataTableRowSelected', '.pdf_menu', '#log_content', '.contentTable', '.infoBoxHeading'];

        $.each(customSelectors, function () {
            if (!$(this).hasClass('gx-container')) {
                $(this).addClass('gx-container');
            }
        });
    });

    /**
     * Normalize tables by custom selectors.
     */
    fixes.push(function () {
        // Append the following array with extra custom selectors.
        var normalizeTables = ['#gm_box_content > table', '#gm_box_content > form > table'];

        $.each(normalizeTables, function () {
            if (!$(this).hasClass('normalize-table')) {
                $(this).addClass('normalize-table');
            }
        });
    });

    /**
     * Add extra classes to the table structure of configuration.php pages.
     */
    fixes.push(function () {
        var tablesArray = $('form[name="configuration"]').children();

        // set $saveBtn only if there is exactly one input[type="submit"]-Button
        if ($('input[type="submit"]').length === 1) {
            var $saveBtn = $('input[type="submit"]');
            $saveBtn.removeClass('button');
            if (!$saveBtn.hasClass('btn')) {
                $saveBtn.addClass('btn');
                $saveBtn.addClass('btn-primary');
            }
        }

        $.each(tablesArray, function (index, element) {
            var labelText = $(element).find('.dataTableContent_gm').first().children().first().text(),
                $elementObj = $(element),
                rightDataTableContent = $($elementObj.find('.dataTableContent_gm')[1]);
            $elementObj.find('tr[bgcolor]').removeAttr('bgcolor');
            $elementObj.find('.dataTableContent_gm').first().addClass('configuration-label');
            $elementObj.find('.dataTableContent_gm').first().children().first().replaceWith(labelText);

            rightDataTableContent.find('br').remove();

            $elementObj.addClass('main-table');

            if (index % 2) {
                $elementObj.addClass('even');
            } else {
                $elementObj.addClass('odd');
            }
        });
        $('.error-logging-select').removeClass('pull-left');
    });

    /**
     * Fixes for the orders table.
     *
     * Some columns swapped or hide, classes was added and some elements will be removed.
     */
    fixes.push(function () {
        var $headingBoxContainer = $('.orders_form'),
            $orderInfoBox = $('#gm_orders');

        $.each($headingBoxContainer.children(), function (index, element) {
            $(element).addClass('hidden');
        });

        $orderInfoBox.addClass('hidden');
    });

    /**
     * Fixes for customer overview page.
     */
    fixes.push(function () {
        var $compatibilityTable = $('.gx-compatibility-table.gx-customer-overview'),
            $pagerRow = $compatibilityTable.find('tr').last();

        $('.info-box').addClass('hidden');
        $('.customer-sort-links').addClass('hidden');
        $pagerRow.find('td').first().parent().parent().parent().appendTo($compatibilityTable.parent());
        $compatibilityTable.find('.arrow-icon').addClass('hidden');
        $compatibilityTable.find('tr').last().remove();

        // Delete guest accounts
        $('#delete-guest-accounts').on('click', function () {
            // Create confirmation dialog
            var $confirmation = $('<div>');
            var $content = $('<span>');

            $content.text(jse.core.lang.translate('CONFIRM_DELETE_GUEST_ACCOUNTS', 'admin_customers'));

            $confirmation.appendTo('body').append($content).addClass('gx-container').dialog({
                'title': jse.core.lang.translate('BUTTON_DELETE_GUEST_ACCOUNTS', 'admin_customers'),
                'modal': true,
                'dialogClass': 'gx-container',
                'buttons': [{
                    'text': jse.core.lang.translate('close', 'buttons'),
                    'class': 'btn',
                    'click': function click() {
                        $(this).dialog('close');
                    }
                }, {
                    'text': jse.core.lang.translate('delete', 'buttons'),
                    'class': 'btn-primary',
                    'click': function click() {
                        $.ajax({
                            url: [window.location.origin + window.location.pathname.replace('customers.php', ''), 'request_port.php', '?module=DeleteGuestAccounts', '&token=' + $('#delete-guest-accounts').data('token')].join(''),
                            type: 'GET',
                            dataType: 'json',
                            data: '',
                            success: function success(p_result_json) {
                                var t_url = window.location.href;
                                if (window.location.search.search('cID=') !== -1) {
                                    t_url = window.location.href.replace(/[&]?cID=[\d]+/g, '');
                                }

                                window.location.href = t_url;

                                return false;
                            }
                        });
                    }
                }],
                'width': 420,
                'closeOnEscape': false,
                'open': function open() {
                    $('.ui-dialog-titlebar-close').hide();
                }
            });
        });
    });

    /**
     * Class fixes for the products and categories overview page.
     */
    fixes.push(function () {
        var $infoBox = $('.gx-categories').find('.info-box'),
            $sortBarRow = $('.dataTableHeadingRow_sortbar'),
            $createNewContainer = $('.create-new-container'),
            pageHeadingElementsArray = $('.pageSubHeading').children(),
            tableCellArray = $('.categories_view_data'),
            $pagerContainer = $('.articles-pager');
        $infoBox.addClass('hidden');
        $sortBarRow.addClass('hidden');
        $.each(tableCellArray, function (index, element) {
            // Replace double '-' with single one.
            var cellObj = $(element);
            if (cellObj.text() === '--') {
                cellObj.text('-');
            }
        });
        $.each(pageHeadingElementsArray, function (index, element) {
            // Page heading actions.
            $(element).addClass('hidden');
        });
        $createNewContainer.removeClass('hidden');

        $.each($pagerContainer.find('.button'), function (index, element) {
            var elementObj = $(element);
            elementObj.addClass('hidden');
            elementObj.removeClass('button');
        });
    });

    /**
     * Add Pagination styles
     */
    fixes.push(function () {
        // Define pagination area where all the pagination stuff is
        var $paginationArea = $this.find('.pagination-control').parents('table:first');

        // Add compatibility classes
        $paginationArea.addClass('gx-container paginator');
    });

    /**
     * Add extra classes to the table structure of configuration.php pages.
     */
    fixes.push(function () {
        var tablesArray = $('form[name="configuration"]').children();
        $.each(tablesArray, function (index, element) {
            var labelText = $(element).find('.dataTableContent_gm').first().children().first().text(),
                $elementObj = $(element),
                rightDataTableContent = $($elementObj.find('.dataTableContent_gm')[1]);
            $elementObj.find('tr[bgcolor]').removeAttr('bgcolor');
            $elementObj.find('.dataTableContent_gm').first().addClass('configuration-label');
            $elementObj.find('.dataTableContent_gm').first().children().first().replaceWith(labelText);

            rightDataTableContent.find('br').remove();

            $elementObj.addClass('main-table');

            if (index % 2) {
                $elementObj.addClass('even');
            } else {
                $elementObj.addClass('odd');
            }
        });
        $('.error-logging-select').removeClass('pull-left');
    });

    /**
     * Change class of all buttons from "button" and "admin_button_green" to "btn"
     */
    fixes.push(function () {
        var selectors = ['.button', '.admin_button_green'];

        $.each(selectors, function () {
            $(this).each(function () {
                if (!$(this).hasClass('btn')) {
                    $(this).addClass('btn');
                }

                $(this).removeClass('button');
                $(this).removeClass('admin_button_green');
            });
        });
    });

    /**
     * Remove img in anchor tags with class btn
     */
    fixes.push(function () {
        $('a.btn').each(function (index, element) {
            if ($(element).find('img').length) {
                $(element).find('img').remove();
            }
        });
    });

    /**
     * Hides an empty container, that takes up space
     */
    fixes.push(function () {
        if (!$('div.orders_form :visible').text().trim().length) {
            $('div.orders_form').parents('table:first').removeProp('cellpadding');
            $('div.orders_form').parents('tr:first').find('br').remove();
            $('div.orders_form').parents('td:first').css('padding', '0');
        }
    });

    /**
     *
     */
    fixes.push(function () {
        $('table.paginator').removeProp('cellspacing').removeProp('cellpadding');
    });

    /**
     * Add extra class for the modal box when a customer group should edit.
     */
    fixes.push(function () {
        var urlHelper = jse.libs.url_arguments,
            // alias
        $form = $('form[name="customers"]');

        if (urlHelper.getCurrentFile() === 'customers.php' && urlHelper.getUrlParameters().action === 'editstatus') {
            $form.find('table').addClass('edit-customer-group-table').attr('cellpadding', '0');
        }
    });

    /**
     * Fix the warning icon element in case a checkbox is next to it
     */
    fixes.push(function () {
        var warningIcon = $('.tooltip_icon.warning');
        if ($(warningIcon).parent().parent().prev('.checkbox-switch-wrapper').length) {
            warningIcon.css('margin-left', '12px');
        }
    });

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Execute the registered fixes.
        $.each(fixes, function () {
            this();
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
